// pages/api/admin/roles/assign.ts
import type { NextApiRequest, NextApiResponse } from "next";
import { prisma } from "@/lib/prisma";
import { requireAdmin } from "@/lib/admin";
import { z } from "zod";

const AssignRoleSchema = z.object({
  userId: z.string().cuid(),
  roleId: z.string().cuid().optional(), // null to remove role assignment
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await requireAdmin(req, res);
  if (!session) return;

  if (req.method !== "POST") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  const parsed = AssignRoleSchema.safeParse(req.body);
  if (!parsed.success) {
    return res.status(400).json({
      ok: false,
      error: "Invalid data",
      details: parsed.error.errors,
    });
  }

  const { userId, roleId } = parsed.data;

  try {
    // Verify user exists
    const user = await prisma.user.findUnique({
      where: { id: userId },
      select: { id: true, email: true, name: true },
    });

    if (!user) {
      return res.status(404).json({ ok: false, error: "User not found" });
    }

    // Verify role exists (if provided)
    if (roleId) {
      const role = await prisma.role.findUnique({
        where: { id: roleId },
        select: { id: true, name: true, isActive: true },
      });

      if (!role) {
        return res.status(404).json({ ok: false, error: "Role not found" });
      }

      if (!role.isActive) {
        return res.status(400).json({
          ok: false,
          error: "Cannot assign inactive role",
        });
      }
    }

    // Update user's custom role
    const updatedUser = await prisma.user.update({
      where: { id: userId },
      data: { customRoleId: roleId || null },
      select: {
        id: true,
        email: true,
        name: true,
        role: true,
        customRole: {
          select: {
            id: true,
            name: true,
            description: true,
          },
        },
      },
    });

    return res.status(200).json({
      ok: true,
      user: updatedUser,
      message: roleId
        ? "Role assigned successfully"
        : "Role assignment removed",
    });
  } catch (error: any) {
    return res.status(500).json({
      ok: false,
      error: "Failed to assign role",
    });
  }
}
